/*
 * Decompiled with CFR 0.152.
 */
package org.dcm4che3.tool.pdf2dcm;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.text.MessageFormat;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;
import javax.xml.parsers.ParserConfigurationException;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.dcm4che3.data.Attributes;
import org.dcm4che3.data.ElementDictionary;
import org.dcm4che3.data.VR;
import org.dcm4che3.io.DicomOutputStream;
import org.dcm4che3.io.SAXReader;
import org.dcm4che3.tool.common.CLIUtils;
import org.dcm4che3.util.StreamUtils;
import org.dcm4che3.util.UIDUtils;
import org.xml.sax.SAXException;

public class Pdf2Dcm {
    private static final ResourceBundle rb = ResourceBundle.getBundle("org.dcm4che3.tool.pdf2dcm.messages");
    private static final long MAX_FILE_SIZE = 0x7FFFFFFEL;
    private static final ElementDictionary DICT = ElementDictionary.getStandardElementDictionary();
    private static final int[] IUID_TAGS = new int[]{0x20000D, 0x20000E, 524312};
    private static final int[] TYPE2_TAGS = new int[]{524323, 524339, 524330};
    private static Attributes staticMetadata;
    private static FileContentType fileContentType;
    private static boolean encapsulatedDocLength;

    public static void main(String[] args) {
        try {
            CommandLine cl = Pdf2Dcm.parseComandLine(args);
            Pdf2Dcm pdf2Dcm = new Pdf2Dcm();
            List argList = cl.getArgList();
            int argc = argList.size();
            if (argc < 2) {
                throw new ParseException(rb.getString("missing"));
            }
            File dest = new File((String)argList.get(argc - 1));
            if ((argc > 2 || new File((String)argList.get(0)).isDirectory()) && !dest.isDirectory()) {
                throw new ParseException(MessageFormat.format(rb.getString("nodestdir"), dest));
            }
            Pdf2Dcm.initialize(cl);
            pdf2Dcm.convert(cl.getArgList());
        }
        catch (ParseException e) {
            System.err.println("pdf2dcm: " + e.getMessage());
            System.err.println(rb.getString("try"));
            System.exit(2);
        }
        catch (Exception e) {
            System.err.println("pdf2dcm: " + e.getMessage());
            e.printStackTrace();
            System.exit(2);
        }
    }

    private static CommandLine parseComandLine(String[] args) throws ParseException {
        Options opts = new Options();
        CLIUtils.addCommonOptions((Options)opts);
        opts.addOption(Option.builder((String)"s").hasArgs().argName("[seq.]attr=value").desc(rb.getString("metadata")).build());
        opts.addOption(Option.builder((String)"f").hasArg().argName("xml-file").desc(rb.getString("file")).build());
        opts.addOption(Option.builder().hasArg().argName("contentType").longOpt("contentType").desc(rb.getString("contentType")).build());
        opts.addOption(Option.builder().longOpt("encapsulatedDocLength").desc(rb.getString("encapsulatedDocLength")).build());
        return CLIUtils.parseComandLine((String[])args, (Options)opts, (ResourceBundle)rb, Pdf2Dcm.class);
    }

    private static FileContentType fileContentType(String s) {
        switch (s.toLowerCase(Locale.ENGLISH)) {
            case "stl": 
            case "model/stl": 
            case "model/x.stl-binary": 
            case "application/sla": {
                return FileContentType.STL;
            }
            case "pdf": 
            case "application/pdf": {
                return FileContentType.PDF;
            }
            case "xml": 
            case "application/xml": {
                return FileContentType.CDA;
            }
            case "mtl": 
            case "model/mtl": {
                return FileContentType.MTL;
            }
            case "obj": 
            case "model/obj": {
                return FileContentType.OBJ;
            }
            case "genozip": 
            case "application/vnd.genozip": {
                return FileContentType.GENOZIP;
            }
            case "vcf.bz2": 
            case "vcfbzip2": 
            case "vcfbz2": 
            case "application/prs.vcfbzip2": {
                return FileContentType.VCF_BZIP2;
            }
            case "boz": 
            case "bz2": 
            case "application/x-bzip2": {
                return FileContentType.DOC_BZIP2;
            }
        }
        throw new IllegalArgumentException(MessageFormat.format(rb.getString("content-type-undetermined"), s));
    }

    private static void initialize(CommandLine cl) throws Exception {
        Pdf2Dcm.createStaticMetadata(cl);
        if (cl.hasOption("contentType")) {
            fileContentType = Pdf2Dcm.fileContentType(cl.getOptionValue("contentType"));
        }
        encapsulatedDocLength = cl.hasOption("encapsulatedDocLength");
    }

    private static void createStaticMetadata(CommandLine cl) throws Exception {
        staticMetadata = new Attributes();
        if (cl.hasOption("f")) {
            staticMetadata = SAXReader.parse((String)cl.getOptionValue("f"));
        }
        CLIUtils.addAttributes((Attributes)staticMetadata, (String[])cl.getOptionValues("s"));
        Pdf2Dcm.supplementMissingUIDs(staticMetadata);
        Pdf2Dcm.supplementType2Tags(staticMetadata);
    }

    private Attributes createMetadata(FileContentType fileContentType, File srcFile) throws Exception {
        Attributes fileMetadata = SAXReader.parse((InputStream)StreamUtils.openFileOrURL((String)fileContentType.getSampleMetadataFile()));
        fileMetadata.addAll(staticMetadata);
        if (!(fileContentType != FileContentType.STL && fileContentType != FileContentType.OBJ || fileMetadata.containsValue(0x200052))) {
            fileMetadata.setString(0x200052, VR.UI, UIDUtils.createUID());
        }
        if (encapsulatedDocLength) {
            fileMetadata.setLong(4325397, VR.UL, new long[]{srcFile.length()});
        }
        return fileMetadata;
    }

    private void convert(List<String> args) throws Exception {
        int argsSize = args.size();
        Path destPath = Paths.get(args.get(argsSize - 1), new String[0]);
        for (String src : args.subList(0, argsSize - 1)) {
            Path srcPath = Paths.get(src, new String[0]);
            if (Files.isDirectory(srcPath, new LinkOption[0])) {
                Files.walkFileTree(srcPath, new Pdf2DcmFileVisitor(srcPath, destPath));
                continue;
            }
            if (Files.isDirectory(destPath, new LinkOption[0])) {
                this.convert(srcPath, destPath.resolve(srcPath.getFileName() + ".dcm"));
                continue;
            }
            this.convert(srcPath, destPath);
        }
    }

    private void convert(Path srcFilePath, Path destFilePath) throws Exception {
        FileContentType fileContentType1 = fileContentType != null ? fileContentType : FileContentType.valueOf(srcFilePath);
        File srcFile = srcFilePath.toFile();
        File destFile = destFilePath.toFile();
        Attributes fileMetadata = this.createMetadata(fileContentType1, srcFile);
        long fileLength = srcFile.length();
        if (fileLength > 0x7FFFFFFEL) {
            throw new IllegalArgumentException(MessageFormat.format(rb.getString("file-too-large"), srcFile));
        }
        try (DicomOutputStream dos = new DicomOutputStream(destFile);){
            dos.writeDataset(fileMetadata.createFileMetaInformation("1.2.840.10008.1.2.1"), fileMetadata);
            dos.writeAttribute(4325393, VR.OB, Files.readAllBytes(srcFile.toPath()));
        }
        System.out.println(MessageFormat.format(rb.getString("converted"), srcFile, destFile));
    }

    private static void supplementMissingUIDs(Attributes metadata) {
        for (int tag : IUID_TAGS) {
            if (metadata.containsValue(tag)) continue;
            metadata.setString(tag, VR.UI, UIDUtils.createUID());
        }
    }

    private static void supplementType2Tags(Attributes metadata) {
        for (int tag : TYPE2_TAGS) {
            if (metadata.contains(tag)) continue;
            metadata.setNull(tag, DICT.vrOf(tag));
        }
    }

    static enum FileContentType {
        PDF("resource:encapsulatedPDFMetadata.xml"),
        CDA("resource:encapsulatedCDAMetadata.xml"),
        STL("resource:encapsulatedSTLMetadata.xml"),
        MTL("resource:encapsulatedMTLMetadata.xml"),
        OBJ("resource:encapsulatedOBJMetadata.xml"),
        GENOZIP("resource:encapsulatedGenozipMetadata.xml"),
        VCF_BZIP2("resource:encapsulatedVCFBzip2Metadata.xml"),
        DOC_BZIP2("resource:encapsulatedDocumentBzip2Metadata.xml");

        private final String sampleMetadataFile;

        public String getSampleMetadataFile() {
            return this.sampleMetadataFile;
        }

        private FileContentType(String sampleMetadataFile) {
            this.sampleMetadataFile = sampleMetadataFile;
        }

        static FileContentType valueOf(Path path) throws IOException {
            String fileName = path.toFile().getName();
            String ext = fileName.substring(fileName.lastIndexOf(46) + 1);
            String contentType = Files.probeContentType(path);
            return Pdf2Dcm.fileContentType(contentType != null ? contentType : ext);
        }
    }

    class Pdf2DcmFileVisitor
    extends SimpleFileVisitor<Path> {
        private final Path srcPath;
        private final Path destPath;

        Pdf2DcmFileVisitor(Path srcPath, Path destPath) {
            this.srcPath = srcPath;
            this.destPath = destPath;
        }

        @Override
        public FileVisitResult visitFile(Path srcFilePath, BasicFileAttributes attrs) throws IOException {
            Path destFilePath = this.resolveDestFilePath(srcFilePath);
            if (!Files.isDirectory(destFilePath, new LinkOption[0])) {
                Files.createDirectories(destFilePath, new FileAttribute[0]);
            }
            try {
                Pdf2Dcm.this.convert(srcFilePath, destFilePath.resolve(srcFilePath.getFileName() + ".dcm"));
            }
            catch (ParserConfigurationException | SAXException e) {
                System.out.println(MessageFormat.format(rb.getString("failed"), srcFilePath, e.getMessage()));
                e.printStackTrace(System.out);
                return FileVisitResult.TERMINATE;
            }
            catch (Exception e) {
                System.out.println(MessageFormat.format(rb.getString("failed"), srcFilePath, e.getMessage()));
                e.printStackTrace(System.out);
            }
            return FileVisitResult.CONTINUE;
        }

        private Path resolveDestFilePath(Path srcFilePath) {
            int srcFilePathNameCount;
            int srcPathNameCount = this.srcPath.getNameCount();
            if (srcPathNameCount == (srcFilePathNameCount = srcFilePath.getNameCount() - 1)) {
                return this.destPath;
            }
            return this.destPath.resolve(srcFilePath.subpath(srcPathNameCount, srcFilePathNameCount));
        }
    }
}

