/*
 * Decompiled with CFR 0.152.
 */
package org.weasis.opencv.op;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.geom.FlatteningPathIterator;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.opencv.core.Core;
import org.opencv.core.CvException;
import org.opencv.core.CvType;
import org.opencv.core.Mat;
import org.opencv.core.MatOfDouble;
import org.opencv.core.MatOfInt;
import org.opencv.core.MatOfPoint;
import org.opencv.core.Point;
import org.opencv.core.Rect;
import org.opencv.core.Scalar;
import org.opencv.core.Size;
import org.opencv.imgcodecs.Imgcodecs;
import org.opencv.imgproc.Imgproc;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.weasis.core.util.FileUtil;
import org.weasis.opencv.data.ImageCV;
import org.weasis.opencv.data.PlanarImage;
import org.weasis.opencv.op.ImageConversion;

public class ImageProcessor {
    private static final Logger LOGGER = LoggerFactory.getLogger(ImageProcessor.class);
    public static final String UNSUPPORTED_SIZE = "Unsupported size: ";

    private ImageProcessor() {
    }

    public static Core.MinMaxLocResult findRawMinMaxValues(PlanarImage img, boolean exclude8bitImage) throws OutOfMemoryError {
        Core.MinMaxLocResult val;
        if (CvType.depth(Objects.requireNonNull(img).type()) <= 1 && exclude8bitImage) {
            val = new Core.MinMaxLocResult();
            val.minVal = 0.0;
            val.maxVal = 255.0;
        } else {
            val = ImageProcessor.findMinMaxValues(img.toMat());
            if (val == null) {
                val = new Core.MinMaxLocResult();
            }
            if (val.minVal == val.maxVal) {
                val.maxVal += 1.0;
            }
        }
        return val;
    }

    public static ImageCV applyLUT(Mat source, byte[][] lut) {
        Mat lutMat;
        Mat srcImg = Objects.requireNonNull(source);
        int lutCh = Objects.requireNonNull(lut).length;
        if (lutCh > 1) {
            lutMat = new Mat();
            ArrayList<Mat> lutList = new ArrayList<Mat>(lutCh);
            for (int i = 0; i < lutCh; ++i) {
                Mat l = new Mat(1, 256, 0);
                l.put(0, 0, lut[i]);
                lutList.add(l);
            }
            Core.merge(lutList, lutMat);
            if (srcImg.channels() < lut.length) {
                Imgproc.cvtColor(srcImg.clone(), srcImg, 8);
            }
        } else {
            lutMat = new Mat(1, 256, CvType.CV_8UC1);
            lutMat.put(0, 0, lut[0]);
        }
        ImageCV dstImg = new ImageCV();
        Core.LUT(srcImg, lutMat, dstImg);
        return dstImg;
    }

    public static ImageCV rescaleToByte(Mat source, double alpha, double beta) {
        ImageCV dstImg = new ImageCV();
        Objects.requireNonNull(source).convertTo(dstImg, 0, alpha, beta);
        return dstImg;
    }

    public static ImageCV invertLUT(ImageCV source) {
        ImageCV dstImg = new ImageCV();
        Core.bitwise_not(Objects.requireNonNull(source), dstImg);
        return dstImg;
    }

    public static ImageCV bitwiseAnd(Mat source, int src2Cst) {
        Objects.requireNonNull(source);
        ImageCV mask = new ImageCV(source.size(), source.type(), new Scalar(src2Cst));
        Core.bitwise_and(source, mask, mask);
        return mask;
    }

    public static ImageCV crop(Mat source, Rectangle area) {
        Objects.requireNonNull(source);
        Rectangle rect = Objects.requireNonNull(area).intersection(new Rectangle(0, 0, source.width(), source.height()));
        if (area.width > 1 && area.height > 1) {
            return ImageCV.toImageCV(source.submat(new Rect(rect.x, rect.y, rect.width, rect.height)));
        }
        return ImageCV.toImageCV(source.clone());
    }

    public static Core.MinMaxLocResult minMaxLoc(RenderedImage source, Rectangle area) {
        ImageCV srcImg = ImageConversion.toMat(Objects.requireNonNull(source), area);
        return Core.minMaxLoc(srcImg);
    }

    public static List<MatOfPoint> transformShapeToContour(Shape shape, boolean keepImageCoordinates) {
        Rectangle b = shape.getBounds();
        if (keepImageCoordinates) {
            b.x = 0;
            b.y = 0;
        }
        ArrayList<MatOfPoint> points = new ArrayList<MatOfPoint>();
        ArrayList<Point> cvPts = new ArrayList<Point>();
        FlatteningPathIterator iterator = new FlatteningPathIterator(shape.getPathIterator(null), 2.0);
        double[] pts = new double[6];
        MatOfPoint p = null;
        while (!iterator.isDone()) {
            int segType = iterator.currentSegment(pts);
            switch (segType) {
                case 0: {
                    ImageProcessor.addSegment(p, cvPts, points);
                    p = new MatOfPoint();
                    cvPts.add(new Point(pts[0] - (double)b.x, pts[1] - (double)b.y));
                    break;
                }
                case 1: 
                case 4: {
                    cvPts.add(new Point(pts[0] - (double)b.x, pts[1] - (double)b.y));
                    break;
                }
            }
            iterator.next();
        }
        ImageProcessor.addSegment(p, cvPts, points);
        return points;
    }

    private static void addSegment(MatOfPoint p, List<Point> cvPts, List<MatOfPoint> points) {
        if (p != null) {
            int last;
            if (cvPts.size() > 1 && cvPts.get((last = cvPts.size() - 1) - 1).equals(cvPts.get(last))) {
                cvPts.remove(last);
            }
            p.fromList(cvPts);
            cvPts.clear();
            points.add(p);
        }
    }

    public static double[][] meanStdDev(Mat source) {
        return ImageProcessor.meanStdDev(source, (Shape)null, null, null);
    }

    public static double[][] meanStdDev(Mat source, Shape shape) {
        return ImageProcessor.meanStdDev(source, shape, null, null);
    }

    public static double[][] meanStdDev(Mat source, Shape shape, Integer paddingValue, Integer paddingLimit) {
        List<Mat> list = ImageProcessor.getMaskImage(source, shape, paddingValue, paddingLimit);
        if (list.size() < 2) {
            return null;
        }
        return ImageProcessor.buildMeanStdDev(list.get(0), list.get(1));
    }

    public static double[][] meanStdDev(Mat source, Mat mask, Integer paddingValue, Integer paddingLimit) {
        if (source == null) {
            return null;
        }
        Mat paddingMask = ImageProcessor.getPixelPaddingMask(source, mask, paddingValue, paddingLimit);
        return ImageProcessor.buildMeanStdDev(source, paddingMask);
    }

    private static double[][] buildMeanStdDev(Mat source, Mat mask) {
        if (source == null) {
            return null;
        }
        MatOfDouble mean = new MatOfDouble();
        MatOfDouble stddev = new MatOfDouble();
        if (mask == null) {
            Core.meanStdDev(source, mean, stddev);
        } else {
            Core.meanStdDev(source, mean, stddev, mask);
        }
        ArrayList<Mat> channels = new ArrayList<Mat>();
        if (source.channels() > 1) {
            Core.split(source, channels);
        } else {
            channels.add(source);
        }
        double[][] val = new double[5][channels.size()];
        for (int i = 0; i < channels.size(); ++i) {
            Core.MinMaxLocResult minMax = mask == null ? Core.minMaxLoc((Mat)channels.get(i)) : Core.minMaxLoc((Mat)channels.get(i), mask);
            val[0][i] = minMax.minVal;
            val[1][i] = minMax.maxVal;
        }
        val[2] = mean.toArray();
        val[3] = stddev.toArray();
        val[4][0] = mask == null ? (double)source.width() * (double)source.height() : (double)Core.countNonZero(mask);
        return val;
    }

    public static List<Mat> getMaskImage(Mat source, Shape shape, Integer paddingValue, Integer paddingLimit) {
        Mat srcImg;
        Objects.requireNonNull(source);
        Mat mask = null;
        if (shape == null) {
            srcImg = source;
        } else {
            Rectangle b = new Rectangle(0, 0, source.width(), source.height()).intersection(shape.getBounds());
            if (b.getWidth() < 1.0 || b.getHeight() < 1.0) {
                return Collections.emptyList();
            }
            srcImg = source.submat(new Rect(b.x, b.y, b.width, b.height));
            mask = Mat.zeros(srcImg.size(), CvType.CV_8UC1);
            List<MatOfPoint> pts = ImageProcessor.transformShapeToContour(shape, false);
            Imgproc.fillPoly(mask, pts, new Scalar(255.0));
        }
        mask = ImageProcessor.getPixelPaddingMask(srcImg, mask, paddingValue, paddingLimit);
        return Arrays.asList(srcImg, mask);
    }

    private static Mat getPixelPaddingMask(Mat source, Mat mask, Integer paddingValue, Integer paddingLimit) {
        if (paddingValue != null && source.channels() == 1) {
            Mat paddingMask;
            if (paddingLimit == null) {
                paddingLimit = paddingValue;
            } else if (paddingLimit < paddingValue) {
                int temp = paddingValue;
                paddingValue = paddingLimit;
                paddingLimit = temp;
            }
            Mat maskPix = new Mat(source.size(), CvType.CV_8UC1, new Scalar(0.0));
            ImageProcessor.excludePaddingValue(source, maskPix, paddingValue, paddingLimit);
            if (mask == null) {
                paddingMask = maskPix;
            } else {
                paddingMask = new ImageCV();
                Core.bitwise_and(mask, maskPix, paddingMask);
            }
            return paddingMask;
        }
        return mask;
    }

    public static Core.MinMaxLocResult minMaxLoc(Mat srcImg, Mat mask) {
        ArrayList<Mat> channels = new ArrayList<Mat>(Objects.requireNonNull(srcImg).channels());
        if (srcImg.channels() > 1) {
            Core.split(srcImg, channels);
        } else {
            channels.add(srcImg);
        }
        Core.MinMaxLocResult result = new Core.MinMaxLocResult();
        result.minVal = Double.MAX_VALUE;
        result.maxVal = -1.7976931348623157E308;
        for (Mat channel : channels) {
            Core.MinMaxLocResult minMax = Core.minMaxLoc(channel, mask);
            result.minVal = Math.min(result.minVal, minMax.minVal);
            if (result.minVal == minMax.minVal) {
                result.minLoc = minMax.minLoc;
            }
            result.maxVal = Math.max(result.maxVal, minMax.maxVal);
            if (result.maxVal != minMax.maxVal) continue;
            result.maxLoc = minMax.maxLoc;
        }
        return result;
    }

    private static void excludePaddingValue(Mat src, Mat mask, int paddingValue, int paddingLimit) {
        Mat dst = new Mat();
        Core.inRange(src, new Scalar(paddingValue), new Scalar(paddingLimit), dst);
        Core.bitwise_not(dst, dst);
        Core.add(dst, mask, mask);
    }

    public static ImageCV scale(Mat source, Dimension dim) {
        if (Objects.requireNonNull(dim).width < 1 || dim.height < 1) {
            throw new IllegalArgumentException(UNSUPPORTED_SIZE + dim);
        }
        ImageCV dstImg = new ImageCV();
        Imgproc.resize(Objects.requireNonNull(source), dstImg, new Size(dim.getWidth(), dim.getHeight()));
        return dstImg;
    }

    public static ImageCV scale(Mat source, Dimension dim, Integer interpolation) {
        if (interpolation == null || interpolation < 0 || interpolation > 4) {
            return ImageProcessor.scale(source, dim);
        }
        if (Objects.requireNonNull(dim).width < 1 || dim.height < 1) {
            throw new IllegalArgumentException(UNSUPPORTED_SIZE + dim);
        }
        ImageCV dstImg = new ImageCV();
        Imgproc.resize(Objects.requireNonNull(source), dstImg, new Size(dim.getWidth(), dim.getHeight()), 0.0, 0.0, interpolation);
        return dstImg;
    }

    public static ImageCV mergeImages(Mat source1, Mat source2, double opacity1, double opacity2) {
        Mat srcImg = Objects.requireNonNull(source1);
        Mat src2Img = Objects.requireNonNull(source2);
        ImageCV dstImg = new ImageCV();
        Core.addWeighted(srcImg, opacity1, src2Img, opacity2, 0.0, dstImg);
        return dstImg;
    }

    private static boolean isGray(Color color) {
        int r = color.getRed();
        return r == color.getGreen() && r == color.getBlue();
    }

    public static ImageCV overlay(Mat source, Mat imgOverlay, Color color) {
        boolean type16bit;
        ImageCV srcImg = ImageCV.toImageCV(Objects.requireNonNull(source));
        Objects.requireNonNull(imgOverlay);
        boolean bl = type16bit = CvType.depth(srcImg.type()) == 2 || CvType.depth(srcImg.type()) == 3;
        if ((type16bit || ImageProcessor.isGray(color)) && srcImg.channels() == 1) {
            int maxColor = Math.max(color.getRed(), Math.max(color.getGreen(), color.getBlue()));
            if (type16bit) {
                int max = CvType.depth(srcImg.type()) == 3 ? Short.MAX_VALUE : 65535;
                maxColor = maxColor * max / 255;
            }
            Mat grayImg = new Mat(srcImg.size(), srcImg.type(), new Scalar(maxColor));
            ImageCV dstImg = new ImageCV();
            srcImg.copyTo(dstImg);
            grayImg.copyTo(dstImg, imgOverlay);
            return dstImg;
        }
        ImageCV dstImg = new ImageCV();
        if (srcImg.channels() < 3) {
            Imgproc.cvtColor(srcImg, dstImg, 8);
        } else {
            srcImg.copyTo(dstImg);
        }
        Mat colorImg = new Mat(dstImg.size(), CvType.CV_8UC3, new Scalar(color.getBlue(), color.getGreen(), color.getRed()));
        double alpha = (double)color.getAlpha() / 255.0;
        if (alpha < 1.0) {
            ImageCV overlay = new ImageCV();
            dstImg.copyTo(overlay);
            Core.copyTo(colorImg, overlay, imgOverlay);
            Core.addWeighted(overlay, alpha, dstImg, 1.0 - alpha, 0.0, dstImg);
        } else {
            colorImg.copyTo(dstImg, imgOverlay);
        }
        return dstImg;
    }

    public static ImageCV overlay(Mat source, RenderedImage imgOverlay, Color color) {
        return ImageProcessor.overlay(source, ImageConversion.toMat(Objects.requireNonNull(imgOverlay)), color);
    }

    public static BufferedImage drawShape(RenderedImage source, Shape shape, Color color) {
        ImageCV srcImg = ImageConversion.toMat(Objects.requireNonNull(source));
        List<MatOfPoint> pts = ImageProcessor.transformShapeToContour(shape, true);
        Imgproc.fillPoly(srcImg, pts, ImageProcessor.getMaxColor(srcImg, color));
        return ImageConversion.toBufferedImage(srcImg);
    }

    public static ImageCV applyCropMask(Mat source, Rectangle b, double alpha) {
        Mat srcImg = Objects.requireNonNull(source);
        ImageCV dstImg = new ImageCV();
        source.copyTo(dstImg);
        b.grow(1, 1);
        if (b.getY() > 0.0) {
            Imgproc.rectangle((Mat)dstImg, new Point(0.0, 0.0), new Point(dstImg.width(), b.getMinY()), new Scalar(0.0), -1);
        }
        if (b.getX() > 0.0) {
            Imgproc.rectangle((Mat)dstImg, new Point(0.0, b.getMinY()), new Point(b.getMinX(), b.getMaxY()), new Scalar(0.0), -1);
        }
        if (b.getX() < (double)dstImg.width()) {
            Imgproc.rectangle((Mat)dstImg, new Point(b.getMaxX(), b.getMinY()), new Point(dstImg.width(), b.getMaxY()), new Scalar(0.0), -1);
        }
        if (b.getY() < (double)dstImg.height()) {
            Imgproc.rectangle((Mat)dstImg, new Point(0.0, b.getMaxY()), new Point(dstImg.width(), dstImg.height()), new Scalar(0.0), -1);
        }
        Core.addWeighted(dstImg, alpha, srcImg, 1.0 - alpha, 0.0, dstImg);
        return dstImg;
    }

    private static Scalar getMaxColor(Mat source, Color color) {
        Scalar scalar;
        boolean type16bit;
        int depth = CvType.depth(source.type());
        boolean bl = type16bit = depth == 2 || depth == 3;
        if (type16bit) {
            int maxColor = Math.max(color.getRed(), Math.max(color.getGreen(), color.getBlue()));
            int max = CvType.depth(source.type()) == 3 ? Short.MAX_VALUE : 65535;
            max = maxColor * max / 255;
            scalar = new Scalar(max);
        } else {
            scalar = new Scalar(color.getBlue(), color.getGreen(), color.getRed());
        }
        return scalar;
    }

    public static ImageCV applyShutter(Mat source, Shape shape, Color color) {
        Mat srcImg = Objects.requireNonNull(source);
        Mat mask = Mat.zeros(srcImg.size(), CvType.CV_8UC1);
        List<MatOfPoint> pts = ImageProcessor.transformShapeToContour(shape, true);
        Imgproc.fillPoly(mask, pts, new Scalar(1.0));
        ImageCV dstImg = new ImageCV(srcImg.size(), srcImg.type(), ImageProcessor.getMaxColor(srcImg, color));
        srcImg.copyTo(dstImg, mask);
        return dstImg;
    }

    public static ImageCV applyShutter(Mat source, RenderedImage imgOverlay, Color color) {
        return ImageProcessor.overlay(source, imgOverlay, color);
    }

    public static ImageCV getRotatedImage(Mat source, int rotateCvType) {
        if (rotateCvType < 0 || rotateCvType > 2) {
            return ImageCV.toImageCV(source.clone());
        }
        Mat srcImg = Objects.requireNonNull(source);
        ImageCV dstImg = new ImageCV();
        Core.rotate(srcImg, dstImg, rotateCvType);
        return dstImg;
    }

    public static ImageCV flip(Mat source, int flipCvType) {
        Objects.requireNonNull(source);
        ImageCV dstImg = new ImageCV();
        Core.flip(source, dstImg, flipCvType);
        return dstImg;
    }

    public static ImageCV warpAffine(Mat source, Mat matrix, Size boxSize, Integer interpolation) {
        Mat srcImg = Objects.requireNonNull(source);
        ImageCV dstImg = new ImageCV();
        if (interpolation == null) {
            interpolation = 1;
        }
        Imgproc.warpAffine(srcImg, dstImg, Objects.requireNonNull(matrix), Objects.requireNonNull(boxSize), interpolation);
        return dstImg;
    }

    public static Core.MinMaxLocResult findMinMaxValues(Mat source) {
        if (source != null) {
            return ImageProcessor.minMaxLoc(source, null);
        }
        return null;
    }

    public static Core.MinMaxLocResult findMinMaxValues(Mat source, Integer paddingValue, Integer paddingLimit) {
        if (source != null) {
            Mat mask = null;
            if (paddingValue != null && source.channels() == 1) {
                mask = new Mat(source.size(), CvType.CV_8UC1, new Scalar(0.0));
                if (paddingLimit == null) {
                    paddingLimit = paddingValue;
                } else if (paddingLimit < paddingValue) {
                    int temp = paddingValue;
                    paddingValue = paddingLimit;
                    paddingLimit = temp;
                }
                ImageProcessor.excludePaddingValue(source, mask, paddingValue, paddingLimit);
            }
            return ImageProcessor.minMaxLoc(source, mask);
        }
        return null;
    }

    public static ImageCV buildThumbnail(PlanarImage source, Dimension iconDim, boolean keepRatio) {
        Objects.requireNonNull(source);
        if (Objects.requireNonNull(iconDim).width < 1 || iconDim.height < 1) {
            throw new IllegalArgumentException(UNSUPPORTED_SIZE + iconDim);
        }
        double scale = Math.min(iconDim.getHeight() / (double)source.height(), iconDim.getWidth() / (double)source.width());
        if (scale >= 1.0) {
            return ImageCV.toImageCV(source.toMat().clone());
        }
        Size dim = keepRatio ? new Size((int)(scale * (double)source.width()), (int)(scale * (double)source.height())) : new Size(iconDim.width, iconDim.height);
        Mat srcImg = Objects.requireNonNull(source).toMat();
        ImageCV dstImg = new ImageCV();
        Imgproc.resize(srcImg, dstImg, dim, 0.0, 0.0, 3);
        return dstImg;
    }

    private static boolean writeExceptReadOnly(File file) {
        return !file.exists() || file.canWrite();
    }

    public static boolean writeImage(Mat source, File file) {
        if (ImageProcessor.writeExceptReadOnly(file)) {
            try {
                return Imgcodecs.imwrite(file.getPath(), source);
            }
            catch (OutOfMemoryError | CvException e) {
                LOGGER.error("Writing Image", e);
                FileUtil.delete(file);
            }
        }
        return false;
    }

    public static boolean writeThumbnail(Mat source, File file, int maxSize) {
        block8: {
            if (ImageProcessor.writeExceptReadOnly(file)) {
                double scale = Math.min((double)maxSize / (double)source.height(), (double)maxSize / (double)source.width());
                if (!(scale < 1.0)) break block8;
                Size dim = new Size((int)(scale * (double)source.width()), (int)(scale * (double)source.height()));
                ImageCV thumbnail = new ImageCV();
                try {
                    Imgproc.resize(source, thumbnail, dim, 0.0, 0.0, 3);
                    MatOfInt map = new MatOfInt(1, 80);
                    boolean bl = Imgcodecs.imwrite(file.getPath(), thumbnail, map);
                    thumbnail.close();
                    return bl;
                }
                catch (Throwable throwable) {
                    try {
                        try {
                            thumbnail.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                        throw throwable;
                    }
                    catch (OutOfMemoryError | CvException e) {
                        LOGGER.error("Writing thumbnail", e);
                        FileUtil.delete(file);
                    }
                }
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean writePNG(Mat source, File file) {
        if (ImageProcessor.writeExceptReadOnly(file)) {
            int channels;
            Mat srcImg = Objects.requireNonNull(source);
            Mat dstImg = null;
            int type = srcImg.type();
            int elemSize = CvType.ELEM_SIZE(type);
            int bpp = elemSize * 8 / (channels = CvType.channels(type));
            if (bpp > 16 || !CvType.isInteger(type)) {
                dstImg = new Mat();
                srcImg.convertTo(dstImg, CvType.CV_16SC(channels));
                srcImg = dstImg;
            }
            try {
                boolean bl = Imgcodecs.imwrite(file.getPath(), srcImg);
                return bl;
            }
            catch (OutOfMemoryError | CvException e) {
                LOGGER.error("", e);
                FileUtil.delete(file);
            }
            finally {
                ImageConversion.releaseMat(dstImg);
            }
        }
        return false;
    }

    public static boolean writeImage(RenderedImage source, File file) {
        if (ImageProcessor.writeExceptReadOnly(file)) {
            ImageCV dstImg = ImageConversion.toMat(source);
            try {
                boolean bl = Imgcodecs.imwrite(file.getPath(), dstImg);
                if (dstImg != null) {
                    dstImg.close();
                }
                return bl;
            }
            catch (Throwable throwable) {
                try {
                    if (dstImg != null) {
                        try {
                            dstImg.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (OutOfMemoryError | CvException e) {
                    LOGGER.error("", e);
                    FileUtil.delete(file);
                }
            }
        }
        return false;
    }

    public static boolean writeImage(Mat source, File file, MatOfInt params) {
        if (ImageProcessor.writeExceptReadOnly(file)) {
            try {
                return Imgcodecs.imwrite(file.getPath(), source, params);
            }
            catch (OutOfMemoryError | CvException e) {
                LOGGER.error("Writing image", e);
                FileUtil.delete(file);
            }
        }
        return false;
    }

    public static ImageCV readImage(File file, List<String> tags) {
        try {
            return ImageProcessor.readImageWithCvException(file, tags);
        }
        catch (OutOfMemoryError | CvException e) {
            LOGGER.error("Reading image", e);
            return null;
        }
    }

    public static ImageCV readImageWithCvException(File file, List<String> tags) {
        Mat img;
        if (!file.canRead()) {
            return null;
        }
        List<String> exifs = tags;
        if (exifs == null) {
            exifs = new ArrayList<String>();
        }
        if ((img = Imgcodecs.imread(file.getPath(), exifs)).width() < 1 || img.height() < 1) {
            throw new CvException("OpenCV cannot read " + file.getPath());
        }
        return ImageCV.toImageCV(img);
    }
}

